import { useState } from "react";
import type { NextPage } from "next";
import Head from "next/head";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";

const CreateAdminPage: NextPage = () => {
  const { t } = useLanguage();
  const router = useRouter();
  const [formData, setFormData] = useState({
    superAdminKey: "",
    name: "",
    email: "",
    password: "",
    confirmPassword: "",
  });
  const [error, setError] = useState("");
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError("");
    setLoading(true);

    // Validate passwords match
    if (formData.password !== formData.confirmPassword) {
      setError(t("signup.passwordsDoNotMatch") || "Passwords do not match");
      setLoading(false);
      return;
    }

    // Validate password length
    if (formData.password.length < 6) {
      setError(
        t("login.passwordMinLength") || "Password must be at least 6 characters"
      );
      setLoading(false);
      return;
    }

    try {
      const response = await fetch("/api/super-admin/create-admin", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          superAdminKey: formData.superAdminKey,
          name: formData.name,
          email: formData.email,
          password: formData.password,
        }),
      });

      const data = await response.json();

      if (!response.ok) {
        setError(
          data.error ||
            t("createAdmin.failedToCreate") ||
            "Failed to create admin user"
        );
      } else {
        setSuccess(true);
        // Redirect to login after 2 seconds
        setTimeout(() => {
          router.push("/login");
        }, 2000);
      }
    } catch (err) {
      setError(
        t("createAdmin.networkError") || "Network error. Please try again."
      );
    } finally {
      setLoading(false);
    }
  };

  if (success) {
    return (
      <div className='min-h-screen flex items-center justify-center bg-gray-50'>
        <Head>
          <title>
            {t("createAdmin.adminCreated")} - Hospital Ticket System
          </title>
        </Head>
        <div className='max-w-md w-full bg-white shadow-lg rounded-lg p-8 text-center'>
          <div className='text-green-500 text-6xl mb-4'>✓</div>
          <h2 className='text-2xl font-bold text-gray-900 mb-2'>
            {t("createAdmin.adminCreated")}
          </h2>
          <p className='text-gray-600 mb-4'>{t("createAdmin.redirecting")}</p>
        </div>
      </div>
    );
  }

  return (
    <div className='min-h-screen flex items-center justify-center bg-gray-50 py-12 px-4 sm:px-6 lg:px-8'>
      <Head>
        <title>{t("createAdmin.title")} - Hospital Ticket System</title>
      </Head>

      <div className='max-w-md w-full space-y-8'>
        <div>
          <h2 className='mt-6 text-center text-3xl font-extrabold text-gray-900'>
            {t("createAdmin.title")}
          </h2>
          <p className='mt-2 text-center text-sm text-gray-600'>
            {t("createAdmin.subtitle")}
          </p>
        </div>

        <form className='mt-8 space-y-6' onSubmit={handleSubmit}>
          <div className='rounded-md shadow-sm -space-y-px'>
            <div>
              <label htmlFor='superAdminKey' className='sr-only'>
                {t("signup.superAdminKey")}
              </label>
              <input
                id='superAdminKey'
                name='superAdminKey'
                type='password'
                required
                className='appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-t-md focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm'
                placeholder={t("signup.superAdminKey")}
                value={formData.superAdminKey}
                onChange={(e) =>
                  setFormData({ ...formData, superAdminKey: e.target.value })
                }
              />
            </div>
            <div>
              <label htmlFor='name' className='sr-only'>
                {t("createAdmin.name")}
              </label>
              <input
                id='name'
                name='name'
                type='text'
                required
                className='appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm'
                placeholder={t("createAdmin.name")}
                value={formData.name}
                onChange={(e) =>
                  setFormData({ ...formData, name: e.target.value })
                }
              />
            </div>
            <div>
              <label htmlFor='email' className='sr-only'>
                {t("login.email")}
              </label>
              <input
                id='email'
                name='email'
                type='email'
                required
                className='appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm'
                placeholder={t("createAdmin.emailAddress")}
                value={formData.email}
                onChange={(e) =>
                  setFormData({ ...formData, email: e.target.value })
                }
              />
            </div>
            <div>
              <label htmlFor='password' className='sr-only'>
                {t("createAdmin.password")}
              </label>
              <input
                id='password'
                name='password'
                type='password'
                required
                className='appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm'
                placeholder={`${t("createAdmin.password")} (min 6 characters)`}
                value={formData.password}
                onChange={(e) =>
                  setFormData({ ...formData, password: e.target.value })
                }
              />
            </div>
            <div>
              <label htmlFor='confirmPassword' className='sr-only'>
                {t("createAdmin.confirmPassword")}
              </label>
              <input
                id='confirmPassword'
                name='confirmPassword'
                type='password'
                required
                className='appearance-none rounded-none relative block w-full px-3 py-2 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-b-md focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 focus:z-10 sm:text-sm'
                placeholder={t("createAdmin.confirmPassword")}
                value={formData.confirmPassword}
                onChange={(e) =>
                  setFormData({ ...formData, confirmPassword: e.target.value })
                }
              />
            </div>
          </div>

          {error && (
            <div className='rounded-md bg-red-50 p-4'>
              <p className='text-sm text-red-800'>{error}</p>
            </div>
          )}

          <div>
            <button
              type='submit'
              disabled={loading}
              className='group relative w-full flex justify-center py-2 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50 disabled:cursor-not-allowed'
            >
              {loading
                ? t("createAdmin.creatingUser")
                : t("createAdmin.createUser")}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default CreateAdminPage;
